//	
//	GhostAPI.H
//	
//	Portable C API for the Ghost Communication Stack.
//	Requires the Ghost basic types definitions and error codes header files.
//	
//	Written by AS
//	Copyright (C) 2001 The LEGO Company. All rights reserved.
//

#ifndef __GHOSTAPI_H__

	#define __GHOSTAPI_H__

	#include "pbktypes.h"
	#include "pbkerror.h"
	#include "pbkexport.h"

	//	Summary:
	//		Stack handle. You obtain this by calling GhCreateStack
	//	
	//	Remarks:
	//		Most functions in the Ghost API take a stack handle as input parameter.
	//		When you don't need it any more, remember to release the stack by 
	//		calling GhDestroyStack.
	//	
	typedef  void   *	GHSTACK;

	//	Summary:
	//		Handle to a command queue. You obtain this by calling GhCreateCommandQueue.
	//	
	//	Remarks:
	//		Once you have a queue handle, you can add commands to a queue (via GhAppendCommand).
	//		Then you can transmit all the commands in the queue by calling either GhDownload or
	//		GhExecute.
	//	
	typedef	 void	*	GHQUEUE;

	//	Summary:
	//		Handle to a single command. You get it by calling GhGetFirstCommand or 
	//		GhGetNextCommand on a queue handle. You use a command handle to get 
	//		specific information on the command (e.g., the reply len and data).
	//	
	//	Example:
	//	
	//	uint32 PBK_STDCALL MyNotifyFunction(	GHNOTIFYCODE		 notifycode, 
	//											PBKRESULT			 pbkresult, 
	//											GHSTACK				 hstack,
	//											GHQUEUE				 hqueue,
	//											GHCOMMAND			 hcommand,
	//											void				*pvcontext			)
	//	{
	//	uint32 ulretcode = 1;
	//	
	//		if ( notifycode == NotifyCommand)
	//		{
	//		uchar szcmd[256], szreply[256];
	//		uint32 ulcmdlen, ulreplylen;
	//		PBKRESULT pbkrc;
	//	
	//			pbkrc = GhGetCommandDataLen( hcommand, &ulcmdlen );
	//			ASSERT( PBK_SUCCEEDED(pbkrc) );
	//			pbkrc = GhGetCommandData( hcommand, szcmd, sizeof(szcmd) );
	//			ASSERT( PBK_SUCCEEDED(pbkrc) );
	//			pbkrc = GhGetCommandReplyLen( hcommand, &ulreplylen );
	//			ASSERT( PBK_SUCCEEDED(pbkrc) );
	//			if ( ulreplylen > 0 )
	//			{
	//				pbkrc = GhGetCommandReply( hcommand, szreply, sizeof(szreply) );
	//				ASSERT( PBK_SUCCEEDED(pbkrc) );
	//			}
	//			
	//			printf("NotifyCommand: ");
	//			if ( ulcmdlen > 0 )
	//				printf("Cmd code: 0x%X ", (uint32)*szcmd );
	//			if ( PBK_SUCCEEDED(pbkresult))
	//			{
	//				printf("Succeeded ");
	//				if ( ulreplylen > 0 )
	//					printf("(Reply: 0x%X)", (uint32)*szreply );
	//			}
	//			else
	//			{
	//				printf("Failed (error code = 0x%X)", pbkresult );
	//	
	//				//	Cause the next commands to be aborted
	//				ulretcode = 0;
	//			}
	//		}
	//		return ulretcode;
	//	}
	//	
	typedef	 void   *	GHCOMMAND;
	

	//	-------------------------------------------------
	//	Constants
	//	

	//	{alias: Notification codes}
	//	Summary:
	//		Notification codes. These codes are passed by the Ghost API to the notification function,
	//		in the ulnotify parameter.
	//	
	typedef enum {
		
			NotifyIgnore	= 0x00,		//	Ignore this notification
			NotifyCommand	= 0x03,		//	Command completed (command sent and answer received) or command error occurred
			NotifyRequest	= 0x04		//	Request completed (the given queue of commands has been completed either successfully or with errors)

	}GHNOTIFYCODE;


	//	-------------------------------------------------
	//	Function prototypes
	//	

	#ifdef __cplusplus
		extern "C" {
	#endif

	//	Callback function
	//	
	//	[TODO] decide whether to keep the STDCALL modifier.
	//	
	typedef uint32 ( PBK_STDCALL *GHNOTIFYFUNCTION)(	GHNOTIFYCODE		 ulnotifycode, 
														PBKRESULT			 pbkresult, 
														GHSTACK				 hstack,
														GHQUEUE				 hqueue,
														GHCOMMAND			 hcommand,
														void				*pvcontext			);

	//	Function Prototypes:
	//	

	//	Create a Ghost communication stack
	//	
	PBKRESULT	PBK_EXPORT GhCreateStack(			const char		*pszport,
													const char		*pszprotocol,
													const char		*pszsession,
													GHSTACK			*phstack			);
						
	//	Destroy a communication stack freeing all its associated memory
	//	
	PBKRESULT	PBK_EXPORT GhDestroyStack(			GHSTACK  hstack				);
										
	//	Find and select the first available device
	//	
	PBKRESULT	PBK_EXPORT GhSelectFirstDevice(
									GHSTACK		hstack,
									char		*pszdevicename,	// OUT
									uint32		 ulbufsize		);
									
	//	Find and select the next available device
	//	
	PBKRESULT	PBK_EXPORT GhSelectNextDevice(		GHSTACK		hstack,
													char		*pszdevicename,	// OUT
													uint32		 ulbufsize		);
								

	//	Return the name of the currently selected device	
	//	
	PBKRESULT	PBK_EXPORT GhGetDeviceName(		GHSTACK		hstack,
												char		*pszdevicename,	// OUT
												uint32		 ulbufsize		);
	
	//	Return the port type of the given stack	
	//	
	PBKRESULT	PBK_EXPORT GhGetPortType(		GHSTACK		hstack,
												char		*psztype,	// OUT
												uint32		 ulbufsize		);

	//	Return the protocol type of the given stack	
	//	
	PBKRESULT	PBK_EXPORT GhGetProtocolType(	GHSTACK		hstack,
												char		*psztype,	// OUT
												uint32		 ulbufsize		);
	
	//	Return the session type of the given stack	
	//	
	PBKRESULT	PBK_EXPORT GhGetSessionType(	GHSTACK		hstack,
												char		*psztype,	// OUT
												uint32		 ulbufsize		);
	
	//	Open the currently selected device
	//	
	PBKRESULT	PBK_EXPORT GhOpen(					GHSTACK hstack );
		
	//	Close the currently selected device
	//	
	PBKRESULT	PBK_EXPORT GhClose(					GHSTACK hstack );
		
	//	Diagnose the communication state
	//	
	PBKRESULT	PBK_EXPORT GhDiagnose(				GHSTACK hstack );


	//	Submit a command queue on the EXECUTE queue
	//	
	PBKRESULT	PBK_EXPORT GhExecute(				GHSTACK		hstack,
													GHQUEUE		hqueue	);
	
	//	Submit a command queue on the DOWNLOAD queue
	//	
	PBKRESULT	PBK_EXPORT GhDownload(				GHSTACK		hstack,
													GHQUEUE		hqueue	);

	//	Submit a command queue on the EXECUTE queue passing in a specific context value
	//	
	PBKRESULT	PBK_EXPORT GhExecuteEx(		GHSTACK		hstack,
											GHQUEUE		hqueue,
											void		*pcontext );
	
	//	Submit a command queue on the DOWNLOAD queue
	//	
	PBKRESULT	PBK_EXPORT GhDownloadEx(	GHSTACK		hstack,
											GHQUEUE		hqueue,
											void		*pcontext );
	
	//	------------------------------------------------------------------
	//	Settings functions
	//	
	
	//	Set the current command interleave between the execute and download queue
	//	
	PBKRESULT	PBK_EXPORT GhSetInterleave(		GHSTACK		hstack, 
												uint32		ulinterleave_execute,
												uint32		ulinterleave_download	);
	
	
	//	Retrieve the currently selected interleave between the execute and download queue
	//	
	PBKRESULT	PBK_EXPORT GhGetInterleave(		GHSTACK		hstack, 
												uint32		*pulinterleave_execute,
												uint32		*pulinterleave_download	);
	
	//	Set the current notification mode to NOWAIT
	//	
	PBKRESULT	PBK_EXPORT GhSetNoWaitMode(	GHSTACK				hstack,
											GHNOTIFYFUNCTION	pfnotify	);
	
	//	Set the current notification mode to NOWAIT
	//	
	PBKRESULT	PBK_EXPORT GhSetWaitMode(	GHSTACK				hstack,
											GHNOTIFYFUNCTION	pfnotify	);
	
	//	Set the number of retries for execute and download commands
	//	
	PBKRESULT	PBK_EXPORT GhSetRetries(	GHSTACK				hstack,
											int32				lretries_execute,
											int32				lnretries_download	);
	
	//	Get the number of retries for execute and download commands
	//	
	PBKRESULT	PBK_EXPORT GhGetRetries(	GHSTACK				hstack,
											int32				*plretries_execute,
											int32				*plretries_download	);
	
	
	
	//	------------------------------------------------------------------
	//	Protocol statistics functions
	//	

	//	Enable statistics collection
	//	(Statistics are disabled by default)
	//	
	PBKRESULT	PBK_EXPORT GhEnableStatistics( GHSTACK		hstack );

	//	Disable statistics collection
	//	(Statistics are disabled by default)
	//	
	PBKRESULT	PBK_EXPORT GhDisableStatistics(	GHSTACK		hstack );
	

	//	Get statistics
	//	
	PBKRESULT	PBK_EXPORT GhGetStatistics(		GHSTACK		hstack,
												uint32		*pulcmdreq,
												uint32		*pulcmdfail,
												uint32		*pulcmdabort,
												uint32		*pultxreq,
												uint32		*pultxfail,
												uint32		*pulrxreq,
												uint32		*pulrxfail
											);


	//	Reset statistics
	//	
	PBKRESULT	PBK_EXPORT GhResetStatistics(		GHSTACK		hstack	);

	//	------------------------------------------------------------------
	//	Timeout settings
	//	

	//	Set new timeout values
	//	
	PBKRESULT	PBK_EXPORT GhSetTimeout(	GHSTACK		hstack,
											uint32		ultimeout_1st,
											uint32		ultimeout_ic
											);

	//	Get current timeout values
	//	
	PBKRESULT	PBK_EXPORT GhGetTimeout(	GHSTACK		hstack,
											uint32		*pultimeout_1st,
											uint32		*pultimeout_ic
											);
	

	//	------------------------------------------------------------------
	//	Device-specific control
	//	
	PBKRESULT	PBK_EXPORT GhDeviceControl(	GHSTACK		 hstack,
											char		*pszcommand,
											char		*pszresult,		
											uint32		 ulresultsize
											);
	
	//	------------------------------------------------------------------
	//	
	//	Queue and Command handling functions
	//	
	
	//	Create a command queue (containing one command to start with) and return handle
	//	
	PBKRESULT	PBK_EXPORT GhCreateCommandQueue( GHQUEUE *phqueue	);

	//	Release a command queue 
	//	
	PBKRESULT	PBK_EXPORT GhDestroyCommandQueue( GHQUEUE hqueue	);

	//	Checks whether a queue has been submitted to Ghost 
	//	
	PBKRESULT	PBK_EXPORT GhCheckQueueInUse( GHQUEUE hqueue, uint32 *pulinuse );

	//	Retrieves the context value associated with the given queue 
	//	
	PBKRESULT	PBK_EXPORT GhGetQueueContext( GHQUEUE hqueue, void **pcontext );	
	
	//	Associates a context value to a given queue 
	//	
	PBKRESULT	PBK_EXPORT GhSetQueueContext( GHQUEUE hqueue, void *pcontext )	;
		

	//	Append a command to the given command queue.
	//	
	PBKRESULT	PBK_EXPORT GhAppendCommand(		GHQUEUE		 hqueue,
												uchar		*pcommanddata,
												uint32		 ulcommandlen,	
												uint32		 ulexpectedreplylen		);
	
	
	//	Get the first command in the queue
	//	
	PBKRESULT	PBK_EXPORT GhGetFirstCommand(		GHQUEUE		hqueue,
													GHCOMMAND	*phcommand	);

	//	Get the next command in the queue
	//	
	PBKRESULT	PBK_EXPORT GhGetNextCommand(		GHCOMMAND	hcommand,
													GHCOMMAND	*phnext		);

	//	Get command data 
	//	
	PBKRESULT	PBK_EXPORT GhGetCommandData(		GHCOMMAND	hcommand,
													uchar		*pcommanddata,		// OUT
													uint32		ulbufsize		);

	//	Get command data length
	//	
	PBKRESULT	PBK_EXPORT GhGetCommandDataLen(		GHCOMMAND	hcommand,
													uint32	    *ulcommandlen 	);
	
	//	Get command reply
	//	
	PBKRESULT	PBK_EXPORT GhGetCommandReply(		GHCOMMAND	hcommand,
													uchar		*preplydata,		// OUT
													uint32		ulbufsize		);

	//	Get command reply length
	//	
	PBKRESULT	PBK_EXPORT GhGetCommandReplyLen(	GHCOMMAND	hcommand,
													uint32	    *ulreplylen 	);

	
	//	-------------------------------------------------------------
	//	
	//	Various Diagnostic/Debugging functions
	//	

	PBKRESULT	PBK_EXPORT GhDiagSelectSpecificPort(	GHSTACK		 hstack,
														char		*pszportname );
	
	PBKRESULT	PBK_EXPORT GhDiagSelectFirstPort(		GHSTACK		 hstack,
														char		*pszportname,	
														uint32		 ulbufsize		);
														
	PBKRESULT	PBK_EXPORT GhDiagSelectNextPort(		GHSTACK		 hstack,
														char		*pszportname,	
														uint32		 ulbufsize		);
	
	#ifdef __cplusplus
		}
	#endif

#endif